package com.androidbook.simplefiles;

import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.Arrays;

import android.app.Activity;
import android.os.Bundle;
import android.util.Log;

public class SimpleFilesActivity extends Activity {
	private static final String DEBUG_TAG = "SimpleFilesActivity Log";
	private static final String SOME_FILE_CONTENTS = "Żółwie morskie można znaleźć na całym świecie, a jednak wszystie ich gatunki znajdują sie na liście zwierząt zagrożonych wyginięciem. Jednym z tych gatunków są żółwie Karetta.";
	private static final String MORE_FILE_CONTENTS = "Na Pacyfiku, żółwie Karetta wykluwają się z jaj na plażach Japonii. Małe żółwiki wychodzą z jaj w nocy, mając nadzieję, że unikną drapieżników i pożarcia. Podążają w kierunku morza kierując się światłem. W przeszłości, tym najjaśniejszym światłem w nocy było samo morze, jednak obecnie oświetlenie używane przez ludzi może zakłócić wędrówkę małych żółwików. Tylko ich części uda się bezpiecznie przejść plażę i dotrzeć do oceanu.";

	/** Metoda wywoływana podczas pierwszego tworzenia aktywności. */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);

		// Tu jest realizowana podstawowa część przykładu - czyli wszystkie operacje na plikach.
		runFileAccessExample();
	}

	public void runFileAccessExample() {
		Log.i(DEBUG_TAG, "Obsługa plików");
		String file1 = "file1.xxx";

		// Jeśli plik istnieje, usuwamy go.
		if (Arrays.binarySearch(fileList(), file1) >= 0) {
			// Usuwamy stary plik.
			deleteFile(file1);
		}

		// Zapis tekstu do pliku, jeśli plik nie istnieje, zostanie utworzony.
		FileOutputStream fos;
		try {
			fos = openFileOutput(file1, MODE_PRIVATE);
			fos.write(SOME_FILE_CONTENTS.getBytes());
			fos.close();
			Log.i(DEBUG_TAG, "Zapisano tekst w pliku: " + file1);
		} catch (Exception e) {
			Log.i(DEBUG_TAG, "Wywołanie metody openFileOutput (nowy plik) zgłosiło wyjątek: "
					+ e.getMessage());
		}

		// Odczyt całej zawartości pliku i wyświetlenie jej w postaci jednego wiersza tekstu.
		readAppFileAndLog(file1);
		// Ponowny odczyt pliku, tym razem w blokach po 70 znaków.
		readAppFileAndLogAsString(file1);

		// Dopisanie do pliku nowego fragmentu tekstu.
		try {
			fos = openFileOutput(file1, MODE_APPEND);
			fos.write(MORE_FILE_CONTENTS.getBytes());
			fos.close();
			Log.i(DEBUG_TAG, "Dodanie treści do pliku: " + file1);
		} catch (Exception e) {
			Log.i(DEBUG_TAG, "Wywołanie metody openFileOutput (dopisanie) zgłosiło wyjątek: "
					+ e.getMessage());
		}

		// Ponowny odczyt całej zawartości pliku, tak byśmy mogli zobaczyć dodany tekst.
		readAppFileAndLog(file1);
		
		Log.i(DEBUG_TAG, "KONTROLA KATALOGU PLIKÓW APLIKACJI przy wykorzystaniu Context.getFilesDir()");
		File pathForAppFiles = getFilesDir();
		logFileDetails(pathForAppFiles);
		
		Log.i(DEBUG_TAG, "Lista plików w katalogu " + pathForAppFiles.getAbsolutePath());
		String[] fileList = pathForAppFiles.list();
		for(int i=0; i< fileList.length; i++)
		{
			Log.i(DEBUG_TAG, "Plik " + i+": " + fileList[i] );
		}

		Log.i(DEBUG_TAG, "KONTROLA KATALOGU PLIKÓW APLIKACJI przy wykorzystaniu  Context.getCacheDir()");
		File pathCacheDir = getCacheDir();
		logFileDetails(pathCacheDir);		

		String strCacheFileName = "myCacheFile.cache";
		File newCacheFile = new File(pathCacheDir, strCacheFileName);
		try {
			Log.i(DEBUG_TAG, "Utworzenie nowego pliku w katalogu pamięci podręcznej (cache): " + strCacheFileName);
			newCacheFile.createNewFile();
			logFileDetails(newCacheFile);
			
			// Zapis do pliku.
			FileOutputStream foCache = new FileOutputStream(newCacheFile.getAbsolutePath());
			foCache.write(SOME_FILE_CONTENTS.getBytes());
			foCache.close();
			
			// Odczyt zawartości zapisanej w pliku.
			Log.i(DEBUG_TAG, "ZAWARTOŚĆ PLIKU Z KATALOUG CACHE:");
			readAnyFileAndLog(newCacheFile.getAbsolutePath());
			
		} catch (Exception e) {
			Log.i(DEBUG_TAG, "Wywołanie metody createNewFile zgłosiło wyjątek: "
					+ e.getMessage());
		}
		
		Log.i(DEBUG_TAG, "Lista plików w katalogu " + pathCacheDir.getAbsolutePath());
		String[] fileListCache = pathCacheDir.list();
		for(int i=0; i< fileListCache.length; i++)
		{
			Log.i(DEBUG_TAG, "Plik " + i+": " + fileListCache[i] );
		}
		
		Log.i(DEBUG_TAG, "Usuwanie plików z katalogu pamięci podręcznej (cache): " + strCacheFileName);
		newCacheFile.delete();
		
		Log.i(DEBUG_TAG, "Lista plików umieszczonych w katalogu " + pathCacheDir.getAbsolutePath());
		String[] fileListCacheRefeshed = pathCacheDir.list();
		for(int i=0; i< fileListCacheRefeshed.length; i++)
		{
			Log.i(DEBUG_TAG, "Plik " + i+": " + fileListCacheRefeshed[i] );
		}
		
		
		// Prezentacja kilku metod obiektu Context
		Log.i(DEBUG_TAG, "Próba usunięcia utworzonego wcześniej pliku: " + file1);
		if (deleteFile(file1)) {
			Log.i(DEBUG_TAG, "Plik " + file1 + " został pomyślnie usunięty.");
		}

		Log.i(DEBUG_TAG, "Koniec przykładowych operacji na plikach.");
	}
	
	// Rejestracja kilku szczegółowych informacji o pliku
	public void logFileDetails(File file)
	{
		if(file.isDirectory())
		{
			Log.i(DEBUG_TAG, file.getAbsolutePath() + " to KATALOG.");			
		}
		if(file.isFile())
		{
			Log.i(DEBUG_TAG, file.getAbsolutePath() + " to PLIK.");			
		}
		if(file.isHidden())
		{
			Log.i(DEBUG_TAG, file.getAbsolutePath() + " jest UKRYTY.");			
		}

		if(file.exists())
		{
			Log.i(DEBUG_TAG, file.getAbsolutePath() + " ISTNIEJE.");			
		}
		if(file.canRead())
		{
			Log.i(DEBUG_TAG, file.getAbsolutePath() + " może być ODCZYTYWANY.");			
		}
		if(file.canWrite())
		{
			Log.i(DEBUG_TAG, file.getAbsolutePath() + " może być ZAPISYWANY.");			
		}	
	}

	// Zapisuje w dzienniku zawartość pliku w blokach po 70 znaków, dodając znak nowego wiersza na końcu każdego bloku.
	// Zazwyczaj takie operacje na plikach są wykonywane w osobnym wątku, jednka w tym przykładzie
	// zależało nam na zachowaniu prostoty i przejrzystości. 
	// Przykład wykonywania operacji na plikach w osobnych wątkach można znaleźć w aplikacji 
	// przykładowej FileStreamOfConsciousness.
	// 
	// Ta metoda może operować wyłącznie na plikach umieszczonych w katalogu plików aplikacji.
	public void readAppFileAndLog(String filename) {

		FileInputStream fis;
		try {
			fis = openFileInput(filename);
			StringBuffer sBuffer = new StringBuffer();
			int chunkSize = 70;
			byte[] bf = new byte[chunkSize];


			// Odczytujemy zawartość w blokach po 50 bajtów.
			while ((fis.read(bf, 0, chunkSize)) != -1) {
				String str = new String(bf);
				sBuffer.append(str + "\n");			
				if(fis.available() < 50)
				{
				Arrays.fill(bf, 0, chunkSize, (byte) ' '); // Zerujemy bufor, dzięki czemu w kolejnym wierszu znajdą się tylko pozostałe bajty.
				}
			}
			fis.close();

			Log.i(DEBUG_TAG, "Zawartość pliku:\n" + sBuffer.toString());
			Log.i(DEBUG_TAG, "\nEOF");
		} catch (Exception e) {
			Log.i(DEBUG_TAG, "Wywołanie metody openFileInput zgłosiło wyjątek: "+ e.getMessage());
		}

	}

	
	// Rejestrujemy w dzienniku zawartośc pliku wyświetlając po 70 znaków w wierszu (co dobrze wygląda w panelu LogCat)
	// Zazwyczaj operacje tego typu są wykonywne w osobnym wątku.
	public void readAnyFileAndLog(String filename) {

		FileInputStream fis;
		try {
			fis = new FileInputStream(filename);
			StringBuffer sBuffer = new StringBuffer();
			int chunkSize = 70;
			byte[] bf = new byte[chunkSize];


			// Odczytujemy po 50 bajtów.
			while ((fis.read(bf, 0, chunkSize)) != -1) {
				String str = new String(bf);
				sBuffer.append(str + "\n");			
				if(fis.available() < 50)
				{
				Arrays.fill(bf, 0, chunkSize, (byte) ' '); // Zerujemy bufor, dzięki czemu w kolejnym wierszu znajdą się tylko pozostałe bajty.
				}
			}


			fis.close();

			Log.i(DEBUG_TAG, "Zawartość pliku:\n" + sBuffer.toString());
			Log.i(DEBUG_TAG, "\nEOF");
		} catch (Exception e) {
			Log.i(DEBUG_TAG, "Wywołanie metody openFileInput zgłosiło wyjątek: "+ e.getMessage());
		}
	}
	
	// Rejestruje w dzienniku zawartość pliku w postaci jednego, długiego łańcucha znaków.
	// Zazwyczaj operacje tego typu są wykonywne w osobnym wątku.
	public void readAppFileAndLogAsString(String filename) {

		FileInputStream fis;
		try {
			fis = openFileInput(filename);
			StringBuffer sBuffer = new StringBuffer();
			DataInputStream dataIO = new DataInputStream(fis);
			String strLine = null;

			// Odczyt zawartości pliku wiersz po wierszu.
			while ((strLine = dataIO.readLine()) != null) {
				sBuffer.append(strLine + "\n");
			}

			dataIO.close();
			fis.close();

			Log.i(DEBUG_TAG, "File Contents:\n" + sBuffer.toString());
			Log.i(DEBUG_TAG, "\nEOF");
		} catch (Exception e) {
			Log.i(DEBUG_TAG, "Wywołanie metody openFileInput zgłosiło wyjątek: "+ e.getMessage());
		}
	}
}